package dev.hydraulic.types.machines;

import java.util.*;

/**
 * Enumerates glibc and muslc, the two C libraries that matter on Linux.
 */
public enum CLibraries implements CLibrary {
    /** The GNU C Library, also known just as "gnu". */
    GLIBC(Set.of("glibc", "gnu")),

    /** The muslc library as used in Alpine Linux, aliased by "musl" and "alpine". */
    MUSLC(Set.of("muslc", "musl", "alpine"));

    private final Set<String> aliases;

    CLibraries(Set<String> aliases) {
        this.aliases = aliases;
    }

    @Override
    public String getDisplayName() {
        return getIdentifier();
    }

    @Override
    public String getIdentifier() {
        return name().toLowerCase(Locale.ROOT);
    }

    @Override
    public Set<String> getAliases() {
        return aliases;
    }

    @Override
    public String getVendor() {
        return "Open source";
    }

    /**
     * An unmodifiable map of lower case names and aliases of well known operating systems.
     */
    public final static Map<String, CLibrary> ALIAS_MAP;

    static {
        var map = new HashMap<String, CLibrary>();
        for (var value : values()) {
            for (var alias : value.aliases) {
                map.put(alias, value);
            }
        }
        ALIAS_MAP = Collections.unmodifiableMap(map);
    }
}
