package dev.hydraulic.types.machines;

import java.util.*;

/**
 * <p>Major CPU architecture families and common names for them.</p>
 *
 * <p>When writing code prefer to use the {@link CPUArchitecture} interface rather than this enum directly. This will allow other people
 * to create new CPU architecture objects and pass them into your code.</p>
 *
 * <p>CPU architectures are included here if they materially affect application binary compatibility at the granularity of a download.
 * Therefore most minor variants such as ISA extensions aren't modelled. Inclusion criteria are similar to those for the
 * {@link OperatingSystems} enum: the CPU must be in general use for computing devices that can have software installed on them.
 * DSPs, proprietary architectures that are used only within a company etc would not be considered, but can still be modelled by
 * creating a custom object that implements {@link CPUArchitecture}. Extinct CPU architectures are also not included, but could be
 * added by a third party library.</p>
 */
public enum CPUArchitectures implements CPUArchitecture {
    /** The classic 32 bit architecture. Vendor is Intel, aliases are x86, i386, x32, x86_32, x86-32. Vendor is Intel. */
    X86("Intel", "x86", "i386", "x32", "x86_32", "x86-32"),

    /** The modern 64 bit version of the x86 architecture. Aliases are amd64, x64, x86-64, x86_64. Vendor is AMD. */
    AMD64("AMD", "amd64", "x64", "x86-64", "x86_64"),

    /** 64 bit ARM as used in modern phones, Apple Silicon etc. Aliases are aarch64, arm64, armv8. */
    AARCH64("ARM", "aarch64", "arm64", "armv8"),

    /** 32 bit ARM as used in older devices. Aliases are arm32, arm. */
    ARM7("ARM", "arm7", "arm32", "arm"),

    // TODO(low): Consider adding hard/soft float variants of ARM, older ARM sub-variants etc.

    /** The 64 bit variant of the open source RISC-V architecture. Aliases are riscv, riscv64, risc-v, vendor is "Open source". */
    RISCV("Open source", "riscv", "riscv64", "risc-v"),

    /** The z/Architecture used in IBM mainframes. */
    ZARCH("IBM", "zarch", "s390", "s390x"),

    /** For representing an unknown architecture, or where the CPU architecture is irrelevant. Vendor is "unknown". Aliases are "unknown" and "other". */
    UNKNOWN("Unknown", "unknown", "other")
    ;

    private final String vendor;
    private final Set<String> aliases;

    CPUArchitectures(String vendor, String... names) {
        this.vendor = vendor;
        aliases = Set.of(names);
    }

    @Override
    public String getDisplayName() {
        return name();
    }

    @Override
    public String getIdentifier() {
        return name().toLowerCase(Locale.ROOT);
    }

    @Override
    public Set<String> getAliases() {
        return aliases;
    }

    @Override
    public String getVendor() {
        return vendor;
    }

    /**
     * An unmodifiable map of lower case names and aliases of well known operating systems.
     */
    public final static Map<String, CPUArchitecture> ALIAS_MAP;

    static {
        var map = new HashMap<String, CPUArchitecture>();
        for (var value : values()) {
            for (var alias : value.aliases) {
                map.put(alias, value);
            }
        }
        ALIAS_MAP = Collections.unmodifiableMap(map);
    }
}
