package dev.hydraulic.types.machines;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static dev.hydraulic.types.machines.CPUArchitectures.*;
import static dev.hydraulic.types.machines.OperatingSystems.*;

/**
 * Represents combined aspects of a computer that can affect binary compatibility.
 * <p>
 * Machines define a small string serialization of the form {@code osname.cpuname}, as generated by {@code toString()} and parsed by {@link
 * #parse(String)}. Machines sort according to their string representation. Subtypes like {@link LinuxMachine} may extend this string
 * encoding to incorporate more detail.
 */
public interface Machine extends Comparable<Machine> {
    /**
     * The operating system, without version information.
     */
    OperatingSystem getOS();

    /**
     * The CPU architecture.
     */
    CPUArchitecture getCPU();

    /**
     * Returns an object that implements Machine and possibly LinuxMachine (defaulting to glibc as the C library and no distribution
     * domain), given the specified operating system and CPU architecture.
     */
    static Machine of(OperatingSystem os, CPUArchitecture cpu) {
        if (os == Linux)
            return new LinuxMachineImpl(cpu, CLibraries.GLIBC, null);
        else
            return new MachineImpl(os, cpu);
    }

    /**
     * Converts a string to a machine spec by parsing it in a lenient manner. The string is split into components using the `.`
     * separator, then the first component is matched against the OS and the second against the CPU architecture. If the OS is Linux
     * then the third (if present) may be a C library and the remaining components are reversed DNS components of the distribution
     * website.
     * <p>
     * Because a Machine must specify a CPU architecture, if the string form doesn't then a default is picked based on the dominant
     * OS type used on that platform, or if no such type is known then [CPUArchitectures.UNKNOWN] is used:
     * <p>
     * - Linux, Windows: AMD64
     * - Android, any Apple OS including macOS: AARCH64
     * - Others: UNKNOWN
     *
     * @throws IllegalArgumentException if any component isn't a recognized name or alias.
     */
    static Machine parse(String spec) {
        String[] components = spec.split(Pattern.quote("."));
        OperatingSystem os = OperatingSystem.from(components[0]);
        CPUArchitecture arch;
        if (components.length > 1) {
            arch = CPUArchitecture.from(components[1]);
        } else {
            if (os == Windows || os == Linux) {
                arch = AMD64;
            } else if (Families.apple().contains(os)) {
                arch = AARCH64;
            } else {
                arch = CPUArchitectures.UNKNOWN;
            }
        }

        if (os == Linux) {
            CLibrary cLibrary = CLibraries.GLIBC;
            if (components.length > 2)
                cLibrary = CLibrary.from(components[2]);

            String distro = null;
            if (components.length > 3)
                distro = Arrays.stream(components).skip(3).collect(Collectors.joining("."));

            return new LinuxMachineImpl(arch, cLibrary, distro);
        }

        if (components.length >= 3)
            throw new IllegalArgumentException("The third component of a machine name is only defined for Linux. You specified: " + spec);

        return new MachineImpl(os, arch);
    }

    /**
     * <p>Converts a string to a machine spec by parsing it in a lenient manner. The string is split into components using the `.`
     * separator, then the first component is matched against the OS and the second against the CPU architecture. If the OS is Linux
     * then the third (if present) may be a C library and the remaining components are reversed DNS components of the distribution
     * website.</p>
     *
     * <p>Unlike {@link #parse(String)}, missing components will be expanded to all known CPU architectures and C libraries. An empty string
     * will yield an empty result. The special string "*" will yield a result with all known operating systems and CPU architectures.</p>
     *
     * <p>If {@code supported} is set to true, combinations that don't represent machines actually in use are excluded.
     * For example {@code parseAll("mac", true)} won't return results with {@link CPUArchitectures#RISCV} included, because (at the time
     * of writing) there is no Mac with a RISC-V chip and probably there never will be. Windows is likewise taken to only support
     * AMD64, x86 and AARCH64, although other combinations have been used in the past. Free UNIXes like Linux, FreeBSD etc are assumed
     * to support every possible CPU architecture, and Linux is assumed to support every CPU architure and C library in combination.
     * Please note that this support matrix isn't guaranteed to be 100% correct, especially as a release of the library ages.</p>
     *
     * @param spec      The input to expand.
     * @param supported Whether to only include combinations that were known to actually be in use at the time of the release of this
     *                  library.
     * @return An unordered set of all known machines that match the input.
     */
    static Set<Machine> parseAll(String spec, boolean supported) {
        Set<Machine> result = new HashSet<>();

        if (spec.equals("*")) {
            for (OperatingSystem os : OperatingSystems.values()) {
                if (os == OperatingSystems.UNKNOWN) continue;
                result.addAll(parseAll(os.getIdentifier(), supported));
            }
            return result;
        }

        String[] components = spec.split(Pattern.quote("."));
        if (components.length == 0)
            return result;

        OperatingSystem os = OperatingSystem.from(components[0]);
        if (components.length == 1) {
            // Add all known CPUs to this OS.
            if (!supported) {
                for (var cpu : CPUArchitectures.values()) {
                    if (cpu == CPUArchitectures.UNKNOWN) continue;
                    result.addAll(parseAll(os.getIdentifier() + "." + cpu.getIdentifier(), supported));
                }
            } else {
                Set<CPUArchitecture> cpus = new HashSet<>();
                if (os == macOS || os == Windows || os == ChromeOS || os == Android) {
                    cpus.add(AMD64);
                    cpus.add(AARCH64);
                } else if (os == Linux || os == FreeBSD || os == OpenBSD || os == NetBSD) {
                    // If there is a CPU Architecture important enough to model in this library, almost certainly, free UNIXes run on it.
                    // https://www.freebsd.org/cgi/man.cgi?query=arch&manpath=FreeBSD+12-current
                    // https://www.openbsd.org/plat.html
                    // http://www.netbsd.org/ports/
                    Collections.addAll(cpus, CPUArchitectures.values());
                    cpus.remove(CPUArchitectures.UNKNOWN);
                } else if (os == zOS) {
                    cpus.add(ZARCH);
                } else if (Families.apple().contains(os)) {
                    cpus.add(AARCH64);
                    // iPhones support older ARM devices.
                    if (os == iOS)
                        cpus.add(ARM7);
                } else if (os == Solaris) {
                    // SPARC isn't modelled by this library because it hasn't been sold for many years.
                    cpus.add(AMD64);
                } else {
                    // For other operating systems they come from outside this library, we just give up.
                    cpus.add(CPUArchitectures.UNKNOWN);
                }
                for (var cpu : cpus)
                    result.addAll(parseAll(os.getIdentifier() + "." + cpu.getIdentifier(), true));
            }
            return result;
        }

        CPUArchitecture cpu = CPUArchitecture.from(components[1]);
        if (components.length == 2) {
            if (os == Linux) {
                // Add all known C libraries.
                for (var cLib : CLibraries.values()) {
                    result.addAll(parseAll(os.getIdentifier() + "." + cpu.getIdentifier() + "." + cLib.getIdentifier(), supported));
                }
            } else {
                result.add(new MachineImpl(os, cpu));
            }
            return result;
        }

        if (os != Linux)
            throw new IllegalArgumentException("The third component of a machine name is only defined for Linux. You specified: " + spec);

        CLibrary cLibrary = CLibrary.from(components[2]);
        String distro = null;
        if (components.length > 3)
            distro = Arrays.stream(components).skip(3).collect(Collectors.joining("."));

        result.add(LinuxMachine.of(cpu, cLibrary, distro));
        return result;
    }

    /**
     * An immutable {@link LinuxMachine} with no distribution domain, AMD64 and glibc.
     */
    LinuxMachine LINUX_AMD64 = LinuxMachine.of(AMD64, CLibraries.GLIBC);

    /**
     * An immutable {@link LinuxMachine} with no distribution domain, AARCH64 and glibc.
     */
    LinuxMachine LINUX_AARCH64 = LinuxMachine.of(AARCH64, CLibraries.GLIBC);

    /**
     * An immutable {@link LinuxMachine} with no distribution domain, AMD64 and muslc.
     */
    LinuxMachine LINUX_AMD64_MUSLC = LinuxMachine.of(AMD64, CLibraries.MUSLC);

    /**
     * An immutable {@link LinuxMachine} with no distribution domain, AARCH64 and muslc.
     */
    LinuxMachine LINUX_AARCH64_MUSLC = LinuxMachine.of(AARCH64, CLibraries.MUSLC);

    /**
     * An immutable {@link Machine} with macOS on AMD64.
     */
    Machine MACOS_AMD64 = of(macOS, AMD64);

    /**
     * An immutable {@link Machine} with macOS on AARCH64.
     */
    Machine MACOS_AARCH64 = of(macOS, AARCH64);

    /**
     * An immutable {@link Machine} with Windows on AMD64.
     */
    Machine WINDOWS_AMD64 = of(Windows, AMD64);

    /**
     * An immutable {@link Machine} with Windows on AARCH64.
     */
    Machine WINDOWS_AARCH64 = of(Windows, AARCH64);

    /**
     * Returns a machine representing the currently executing JVM's environment, based on reading system properties and file paths.
     * This is currently capable of detecting Windows, Mac and Linux but other operating systems will be detected as
     * {@link OperatingSystems#UNKNOWN}. The C library for Linux hosts is detected by looking for {@code /lib/libc.musl-x86_64.so.1}.
     */
    static Machine current() {
        boolean isWindows = System.getProperty("os.name").startsWith("Windows");
        boolean isMac = System.getProperty("os.name").toLowerCase(Locale.ROOT).startsWith("mac");
        boolean isLinux = System.getProperty("os.name").toLowerCase(Locale.ROOT).startsWith("linux");

        OperatingSystem os;
        if (isWindows)
            os = Windows;
        else if (isMac)
            os = macOS;
        else if (isLinux)
            os = Linux;
        else
            os = OperatingSystems.UNKNOWN;

        String cpuArch = System.getProperty("os.arch");
        if (cpuArch == null)
            throw new NullPointerException("The os.arch system property is not available.");
        var cpu = CPUArchitecture.fromOrUnknown(cpuArch);

        if (os == Linux) {
            CLibrary cLib = CLibraries.GLIBC;
            if (Files.exists(Path.of("/lib/libc.musl-x86_64.so.1")))
                cLib = CLibraries.MUSLC;
            return LinuxMachine.of(cpu, cLib);
        } else {
            return Machine.of(os, cpu);
        }
    }
}
