package dev.hydraulic.types.mimetyped;

import org.jetbrains.annotations.NotNull;

import java.util.Objects;

/**
 * A value-based type that wraps an object (byte array, byte buffer, stream, or any arbitrary object) with an associated MIME type or
 * MIME type registry.
 *
 * @param <T> The type of object that is wrapped.
 * @param <M> A marker type that isn't represented or stored at runtime. It exists only for source documentation, type inference, reflection
 *            and languages that have extension functions. So, feel free to use a wildcard type here, you won't lose anything except some
 *            type safety.
 */
public final class MimeTypedObject<T, M extends MimeType> implements MimeTyped<T, M> {
    @NotNull
    private final T content;

    @NotNull
    private final String mimeType;

    /** application/octet-stream */
    public static <T> MimeTypedObject<T, MimeType.application.octetStream> octetStream(T content) {
        return new MimeTypedObject<>(content, "application/octet-stream");
    }

    /** image/subtype */
    public static <T> MimeTypedObject<T, MimeType.image> image(T content, String subtype) {
        return new MimeTypedObject<>(content, "image/" + subtype);
    }

    /** application/subtype */
    public static <T> MimeTypedObject<T, MimeType.application> application(T content, String subtype) {
        return new MimeTypedObject<>(content, "application/" + subtype);
    }

    static {
        MimeTypedObject<byte[], MimeType.image> s = MimeTypedObject.image(new byte[]{1,2,3}, "gif");
    }

    /**
     * Constructs an immutable mime-typed string.
     *
     * @param content  The wrapped string.
     * @param mimeType A MIME type.
     */
    public MimeTypedObject(T content, String mimeType) {
        assert mimeType.length() > 0;
        this.content = Objects.requireNonNull(content);
        this.mimeType = Objects.requireNonNull(mimeType);
    }

    /** The MIME type. */
    public String mimeType() {
        return mimeType;
    }

    /** The wrapped content. */
    public T content() {
        return content;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        MimeTypedObject<?, ?> mimeTyped = (MimeTypedObject<?, ?>) o;
        return content.equals(mimeTyped.content) && mimeType.equals(mimeTyped.mimeType);
    }

    @Override
    public int hashCode() {
        return Objects.hash(content, mimeType);
    }

    /** Returns {@code [mime/type] content}.*/
    @Override
    public String toString() {
        return "[" + mimeType + "] " + content;
    }
}
