package dev.hydraulic.types.mimetyped;

import org.intellij.lang.annotations.Language;
import org.jetbrains.annotations.NotNull;

import java.io.Serializable;
import java.util.Comparator;
import java.util.Objects;
import java.util.stream.IntStream;

/**
 * A value-based type that combines a string with an associated MIME type. The class implements {@link CharSequence} by delegation, thus
 * can be treated as if it were the underlying string in some cases. Two instances are compared using {@link #MIME_TYPE_THEN_CONTENT}.
 *
 * @param <M> A marker type that isn't represented or stored at runtime. It exists only for source documentation, type inference, reflection
 *            and languages that have extension functions. So, feel free to use a wildcard type here, you won't lose anything except some
 *            type safety.
 */
public final class MimeTypedString<M extends MimeType> implements CharSequence, Serializable, Comparable<MimeTypedString<M>>, MimeTyped<String, M> {
    private final String content;
    private final String mimeType;

    private static final long serialVersionUID = 5289920660193190558L;

    /** Returns the text as {@code text/subtype}. */
    public static MimeTypedString<MimeType.text> text(String subtype, String text) {
        assert subtype.length() > 0;
        return new MimeTypedString<>(text, "text/" + subtype);
    }

    /** {@code text/plain} */
    public static MimeTypedString<MimeType.text.plain> plain(String text) {
        return new MimeTypedString<>("plain", text);
    }

    /** {@code application/xml} */
    public static MimeTypedString<MimeType.application.xml> xml(@Language("xml") String xml) {
        return new MimeTypedString<>(xml, "application/xml");
    }

    /** {@code application/json} */
    public static MimeTypedString<MimeType.text.json> json(@Language("json") String json) {
        return new MimeTypedString<>(json, "application/json");
    }

    /** {@code text/html} */
    public static MimeTypedString<MimeType.text.html> html(@Language("html") String html) {
        return new MimeTypedString<>("html", html);
    }

    /** {@code text/csv} */
    public static MimeTypedString<MimeType.text.csv> csv(@Language("csv") String csv) {
        return new MimeTypedString<>("csv", csv);
    }

    /** {@code text/markdown} */
    public static MimeTypedString<MimeType.text.markdown> markdown(@Language("Markdown") String text) {
        return new MimeTypedString<>(text, "text/markdown");
    }

    /**
     * Constructs an immutable mime-typed string.
     *
     * @param content The wrapped string.
     * @param mimeType A MIME type.
     */
    public MimeTypedString(String content, String mimeType) {
        this.content = content;
        this.mimeType = mimeType;
    }

    /** The wrapped content. */
    public String content() {
        return content;
    }

    /** The MIME type. */
    public String mimeType() {
        return mimeType;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        MimeTypedString<?> that = (MimeTypedString<?>) o;
        return content.equals(that.content) && mimeType.equals(that.mimeType);
    }

    @Override
    public int hashCode() {
        return Objects.hash(content, mimeType);
    }

    /** Returns {@link #content}. */
    @Override
    public String toString() {
        return content;
    }

    @Override
    public int length() {
        return content.length();
    }

    @Override
    public char charAt(int index) {
        return content.charAt(index);
    }

    @Override
    @NotNull
    public CharSequence subSequence(int start, int end) {
        return content.subSequence(start, end);
    }

    @Override
    @NotNull
    public IntStream chars() {
        return content.chars();
    }

    @Override
    @NotNull
    public IntStream codePoints() {
        return content.codePoints();
    }

    @Override
    public int compareTo(@NotNull MimeTypedString o) {
        return MIME_TYPE_THEN_CONTENT.compare(this, o);
    }

    /**
     * A comparator that sorts first by mime type, and then by content.
     */
    public static Comparator<MimeTypedString<?>> MIME_TYPE_THEN_CONTENT = Comparator.comparing((MimeTypedString<?> a) -> a.mimeType).thenComparing((MimeTypedString<?> a) -> a.content);
}
