package hydraulic.types.progress

import hydraulic.types.progress.Progress.Units.BYTES

/**
 * Exposes the progress of some operation.
 *
 * Implementations may be simple immutable holder objects, mutable holders or engine-like objects that expose progress with implementation
 * defined mutation points.
 *
 * # Properties
 *
 * [completed] and [expectedTotal] are measured in terms of the specific [units]. By default, a work report has no message, `completed` = 0
 * and `expectedTotal` = 1, which is a special state considered [indeterminate]. Typically, a progress tracker will react to such a report
 * by showing some sort of spinner animation to communicate that something is being done. If [expectedTotal] is higher than 1 then it is
 * considered to be a measurable operation and a progress bar of some sort is a likely implementation.
 *
 * It's valid for [completed] to go backwards, for [expectedTotal] to change including by going backwards, and for deliveries of this object
 * to be interleaved with reports without quantities i.e. if progress stops being possible to report for some reason.
 *
 * [completed] becoming equal to [expectedTotal] implies the operation has completed and should cause rendered progress to disappear. If you
 * want to express the notion that a progress operation has gone 'indeterminate' (e.g. finished but not really finished), send another
 * report with [completed] = 0 and [expectedTotal] = 1.
 *
 * # Mutability
 *
 * Although this interface doesn't define any way to mutate a progress report, implementations of this interface may be mutable or even the
 * actual object doing the work. This is for performance reasons, to avoid needing to create garbage if reporting progress in a tight loop.
 * In such a case the resolution of progress tracking should be defined by the consumer of the reports (see [Progress.Consumer]).
 *
 * For the duration of a [Progress.Consumer.accept] call a report isn't allowed to change, but if you wish to keep a progress report around
 * beyond the scope of a delivery then use [immutableReport], which is specified to return an immutable object exposing the same
 * data as this one.
 */
interface Progress {
    /**
     * String intended to be translated into the user's local language but may be a language-independent message suitable for printing next
     * to a progress indicator of some sort, e.g. a URL or file name. If null, UI should use a generic message like "Working".
     */
    val message: String?

    // TODO(low): Better localization support.

    /** The current belief about the total units for the entire operation (i.e. not how much remains). */
    val expectedTotal: Long

    /**
     * Units of work done so far in the operation.
     */
    val completed: Long

    /**
     * What units [completed] and [expectedTotal] are measured in.
     */
    val units: Units

    /**
     * Returns true if [expectedTotal] is 1 and [completed] is zero or one, which indicates that this operation should be treated as of
     * indeterminate length.
     */
    val indeterminate: Boolean get() = expectedTotal == 1L && completed <= 1L

    /** Returns true when [completed] >= [expectedTotal]. */
    val complete: Boolean get() = completed >= expectedTotal

    /**
     * An ordered list of progress reports that provide further information on the progress of the current step represented by this report.
     *
     * Sub-reports provide a way to express hierarchical progress. Although deeply nested progress reports aren't very useful
     * for visualization and will usually be truncated at one or two levels deep before display, they arise naturally when composing
     * libraries. There's no obligation for a progress tracker to consume sub-reports.
     *
     * Positioning within this list is meaningful and may be used by progress trackers to provide visual stability. It's OK for entries
     * to be null: this allows an operation to allocate an array up front into which multiple parallel sub-tasks write their progress
     * reports, with null indicating a task that either hasn't started yet or which has completed. An entry should be nulled out *after*
     * sending a completion report in that case.
     */
    val subReports: List<Progress?>

    /**
     * Returns an immutable copy of the progress report. Note that this may well be a different object if the implementation of
     * [Progress] is 'heavy'.
     */
    fun immutableReport(): ImmutableProgress

    /**
     * What the units in a report represent. If a report is made with [BYTES] as the unit, some progress trackers may calculate
     * an average speed.
     */
    enum class Units {
        /** "Work units" that don't have any particular meaning. */
        ABSTRACT,

        /** Work that's measured in bytes. */
        BYTES

        // TODO(low): Add a distinction between consistent and inconsistent abstract work units. Consistent work units will have ETAs calculated.
    }

    /** An object that generates progress reports. */
    interface Trackable {
        /**
         * Sets the progress tracker to be used.
         *
         * Progress reports may be passed to the callback in parallel. Implementations will normally narrow the return type to allow for
         * fluent usage.
         */
        fun trackProgressWith(consumer: Progress.Consumer): Trackable   // TODO(refactor): Make this a subscription type interface.
    }

    /**
     * A specialization of [Consumer] that accepts progress reports.
     */
    @FunctionalInterface
    fun interface Consumer : java.util.function.Consumer<Progress> {
        /**
         * Receives a progress report.
         *
         * Implementations of this interface should be fast, as performance sensitive operations might call [accept] from hot loops.
         * Implementations can assume the report is immutable _only for the duration of the [accept] call_, as the caller is allowed to use
         * a mutable report and reuse it after it's been accepted.
         */
        override fun accept(progress: Progress)
    }
}
