package dev.hydraulic.types.machines;

import java.util.Locale;

/**
 * <p>Models a C library for platforms that have multiple competing implementations.</p>
 *
 * <p>Currently this means just Linux. We don't try to model the varying implementations of the Microsoft Visual C++ runtime because
 * it's usually a bundled part of the app and is thus not relevant for modelling application compatibility. Other platforms tend to
 * have a single C runtime implementation.</p>
 */
public interface CLibrary extends MachineCompatibilityAspect {
    /**
     * Lower-cases the name parameter and then looks it up in the {@link CLibraries#ALIAS_MAP alias map}.
     *
     * @param name The primary name or alias of an operating system. Case-insensitive.
     * @throws IllegalArgumentException if name is blank.
     * @return a member of the enum or null if not recognized.
     */
    static CLibrary fromOrNull(String name) {
        if (name.isBlank())
            throw new IllegalArgumentException("name is blank");
        return CLibraries.ALIAS_MAP.get(name.toLowerCase(Locale.ROOT));
    }

    /**
     * Returns a C library from the {@link CLibraries#ALIAS_MAP alias map} or throws {@link UnrecognizedAliasException}.
     *
     * @param name primary name or alias of the well known operating system. Case-insensitive.
     * @throws IllegalArgumentException if name is blank.
     * @throws UnrecognizedAliasException if name isn't a known operating system name or alias (compared case insensitively).
     * @return the C library enum value.
     */
    public static CLibrary from(String name) {
        var lib = fromOrNull(name);
        if (lib == null)
            throw new UnrecognizedAliasException(name, CLibraries.ALIAS_MAP.keySet());
        return lib;
    }
}
