package dev.hydraulic.types.machines;

import java.util.Objects;

/**
 * Models a machine that runs Linux specifically, adding the additional detail of which C library and distribution it uses. In future
 * additional information may be added here if relevant to application compatibility.
 */
public interface LinuxMachine extends Machine {
    /**
     * Which type of C Library (glibc or muslc) this machine uses.
     */
    CLibrary getCLibrary();

    /**
     * The reversed domain name of the primary website used by this Linux distribution project, or null if unknown/irrelevant.
     * <p>
     * Rather than try to define a custom namespace to identify Linux distributions, we simply defer to DNS and assume that all
     * distributions published via a particular website domain are sufficiently similar to generalize over. Sub-versions are not
     * considered here - versioning is currently not expressed by this set of types.
     * <p>
     * Examples: `com.ubuntu`, `org.getfedora`, `org.debian` etc.
     */
    String getDistributionReverseDNSName();

    /**
     * Returns an immutable object holding the given values.
     *
     * @param cpu The CPU architecture, must not be null.
     * @param cLibrary The C library, must not be null.
     * @param distributionReverseDNSName No default, may be null.
     */
    static LinuxMachine of(CPUArchitecture cpu, CLibrary cLibrary, String distributionReverseDNSName) {
        return new LinuxMachineImpl(
                Objects.requireNonNull(cpu),
                Objects.requireNonNull(cLibrary),
                distributionReverseDNSName
        );
    }

    /**
     * Returns an immutable object holding the given values or defaults and no distributiond domain.
     *
     * @param cpu The CPU architecture, must not be null.
     * @param cLibrary The C library, must not be null.
     */
    static LinuxMachine of(CPUArchitecture cpu, CLibrary cLibrary) {
        return of(cpu, cLibrary, null);
    }

    /**
     * Returns an immutable object of the given CPU architecture and defaulting to glibc / no distribution domain.
     * @param cpu The CPU architecture, must not be null.
     */
    static LinuxMachine of(CPUArchitecture cpu) {
        return of(cpu, null, null);
    }
}
