/**
 * <p>Types for modelling aspects of a computer relevant to the execution of native code, such as the operating system, CPU architecture and
 * C library.</p>
 *
 * Each aspect of machine compatibility is modelled with an interface that extends
 * {@link dev.hydraulic.types.machines.MachineCompatibilityAspect} and an enum that contains well known implementations of that aspect.
 * For example the {@link dev.hydraulic.types.machines.CPUArchitecture} interface is implemented by each member of the
 * {@link dev.hydraulic.types.machines.CPUArchitectures} enum which contains values for AMD64, AARCH64 and so on.
 *
 * <p>This package is only intended to contain types, not functionality. If you're looking for ways to get information about the operating
 * system you're running on, check out <a href="https://oshi.github.io/oshi/">OSHI</a>.</p>
 *
 * <h2>Comprehensiveness</h2>
 *
 * <p>The primary goal of this package is to provide a schema for representing hardware and software in active use as of January 2022 for
 * general purpose computing, whilst still allowing extension for more exotic components that maybe aren't widely used (see below).</p>
 *
 * <p>Extinct or hobby components won't be included unless someone can provide a compelling use case, or argue convincingly that they've
 * changed status ("hobby" is defined as being developed for its own sake). Specialized embedded systems that aren't intended for general
 * purpose computing also don't fall under this definition. Variants so minor that they can be binary compatible from the perspective
 * of application developers aren't distinguished from each other.</p>
 *
 * <h2>Naming and aliases</h2>
 *
 * <p>Enumeration values have names that match the capitalization used by the vendors, not the normal enum capitalization rules for Java.
 * This is to make source code read more naturally. If you want to put a name into a user interface, use
 * {@link dev.hydraulic.types.machines.MachineCompatibilityAspect#getDisplayName()} which returns the brand name including characters that
 * aren't legal in Java identifiers.</p>
 *
 * <p>Each interface contains a {@code from} method (and variants) that parse a variety of common aliases for each aspect. This is intended
 * to make handling user input and file name parsing more robust, whilst avoiding the question of what a given aspect 'should' be called.
 * This is especially useful for macOS which is known by a variety of names (like {@code mac}, {@code macos}, {@code osx} and
 * {@code darwin}), and also CPU architectures. Aliases can be found using the static
 * {@code ALIAS_MAP} on each enum, or by calling {@link dev.hydraulic.types.machines.MachineCompatibilityAspect#getAliases()} for a given
 * entry, and the sets returned will include the lower cased name of the enum value itself.</p>
 *
 * <h2>Forwards/backwards compatibility</h2>
 *
 * <p>The enums are intended to be as exhaustive as possible within the rules given above. Over time new components are developed and
 * released, and these will be added to new versions of the library. Thus if you write code that de-serializes these enums you should be
 * able to handle unrecognized values from the future by mapping them to null or the entry {@code Other}. {@link java.io.ObjectInputStream}
 * doesn't support this, so consider wrapping types if you need to serialize enums with Java serialization in ways that need to be forwards
 * compatible.</p>
 *
 * <p>Other rules are:</p>
 *
 * <ul>
 * <li>No entries will ever be removed, even if they become extinct.</li>
 * <li>Operating systems may be added or removed from the family sets ({@code apple}, {@code linuxKernelBased}, {@code unix} etc), although
 * it's unlikely anything about them would change enough to justify this.</li>
 * <li>Aliases may be added but will never be removed.</li>
 * <li>Vendors and display names may change at any time, e.g. to track rebrandings.</li>
 * </ul>
 *
 * <h2>Extension</h2>
 *
 * It's impossible and/or undesirable to list every possible CPU architecture and operating system in an enum. If a component you want isn't
 * available but you need to express it to a library that uses this API, implement the interfaces
 * {@link dev.hydraulic.types.machines.CPUArchitecture} or {@link dev.hydraulic.types.machines.OperatingSystem} yourself. They don't contain
 * any functionality, only naming related methods intended to enable interoperability and user interface generation.
 *
 * If you need to model an aspect of compatibility that goes beyond what this API does, e.g. individual Linux distributions, consider
 * creating an interface that extends one of the interfaces here. You can then delegate to the base enum values for the base methods and
 * only implement the extra properties you need.
 */
package dev.hydraulic.types.machines;
