package dev.hydraulic.types.machines;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Locale;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * Represents combined aspects of a computer that can affect binary compatibility.
 *
 * Machines define a small string serialization of the form {@code osname.cpuname}, as generated by {@code toString()} and parsed by {@link
 * #parse(String)}. Machines sort according to their string representation. Subtypes like {@link LinuxMachine} may extend this string
 * encoding to incorporate more detail.
 */
public interface Machine extends Comparable<Machine> {
    /**
     * The operating system, without version information.
     */
    OperatingSystem getOS();

    /**
     * The CPU architecture.
     */
    CPUArchitecture getCPU();

    /**
     * Returns an object that implements Machine and possibly LinuxMachine (defaulting to glibc as the C library and no distribution
     * domain), given the specified operating system and CPU architecture.
     */
    static Machine of(OperatingSystem os, CPUArchitecture cpu) {
        if (os == OperatingSystems.Linux)
            return new LinuxMachineImpl(cpu, CLibraries.GLIBC, null);
        else
            return new MachineImpl(os, cpu);
    }

    /**
     * Converts a string to a machine spec by parsing it in a lenient manner. The string is split into components using the `.`
     * separator, then the first component is matched against the OS and the second against the CPU architecture. If the OS is Linux
     * then the third (if present) may be a C library and the remaining components are reversed DNS components of the distribution
     * website.
     *
     * Because a Machine must specify a CPU architecture, if the string form doesn't then a default is picked based on the dominant
     * OS type used on that platform, or if no such type is known then [CPUArchitectures.UNKNOWN] is used:
     *
     * - Linux, Windows: AMD64
     * - Android, any Apple OS including macOS: AARCH64
     * - Others: UNKNOWN
     *
     * @throws IllegalArgumentException if any component isn't a recognized name or alias.
     */
    static Machine parse(String spec) {
        String[] components = spec.split(Pattern.quote("."));
        OperatingSystem os = OperatingSystem.from(components[0]);
        CPUArchitecture arch;
        if (components.length > 1) {
            arch = CPUArchitecture.from(components[1]);
        } else {
            if (os == OperatingSystems.Windows || os == OperatingSystems.Linux) {
                arch = CPUArchitectures.AMD64;
            } else if (OperatingSystems.Families.apple().contains(os)) {
                arch = CPUArchitectures.AARCH64;
            } else {
                arch = CPUArchitectures.UNKNOWN;
            }
        }

        if (os == OperatingSystems.Linux) {
            CLibrary cLibrary = CLibraries.GLIBC;
            if (components.length > 2)
                cLibrary = CLibrary.from(components[2]);

            String distro = null;
            if (components.length > 3)
                distro = Arrays.stream(components).skip(3).collect(Collectors.joining("."));

            return new LinuxMachineImpl(arch, cLibrary, distro);
        }

        if (components.length >= 3)
            throw new IllegalArgumentException("The third component of a machine name is only defined for Linux. You specified: $spec");

        return new MachineImpl(os, arch);
    }

    /** An immutable {@link LinuxMachine} with no distribution domain, AMD64 and glibc. */
    LinuxMachine LINUX_AMD64 = LinuxMachine.of(CPUArchitectures.AMD64, CLibraries.GLIBC);

    /** An immutable {@link LinuxMachine} with no distribution domain, AARCH64 and glibc. */
    LinuxMachine LINUX_AARCH64 = LinuxMachine.of(CPUArchitectures.AARCH64, CLibraries.GLIBC);

    /** An immutable {@link LinuxMachine} with no distribution domain, AMD64 and muslc. */
    LinuxMachine LINUX_AMD64_MUSLC = LinuxMachine.of(CPUArchitectures.AMD64, CLibraries.MUSLC);

    /** An immutable {@link LinuxMachine} with no distribution domain, AARCH64 and muslc. */
    LinuxMachine LINUX_AARCH64_MUSLC = LinuxMachine.of(CPUArchitectures.AARCH64, CLibraries.MUSLC);

    /** An immutable {@link Machine} with macOS on AMD64. */
    Machine MACOS_AMD64 = of(OperatingSystems.macOS, CPUArchitectures.AMD64);

    /** An immutable {@link Machine} with macOS on AARCH64. */
    Machine MACOS_AARCH64 = of(OperatingSystems.macOS, CPUArchitectures.AARCH64);

    /** An immutable {@link Machine} with Windows on AMD64. */
    Machine WINDOWS_AMD64 = of(OperatingSystems.Windows, CPUArchitectures.AMD64);

    /** An immutable {@link Machine} with Windows on AARCH64. */
    Machine WINDOWS_AARCH64 = of(OperatingSystems.Windows, CPUArchitectures.AARCH64);

    /**
     * Returns a machine representing the currently executing JVM's environment, based on reading system properties and file paths.
     * This is currently capable of detecting Windows, Mac and Linux but other operating systems will be detected as
     * {@link OperatingSystems#UNKNOWN}. The C library for Linux hosts is detected by looking for {@code /lib/libc.musl-x86_64.so.1}.
     */
    static Machine current() {
        boolean isWindows = System.getProperty("os.name").startsWith("Windows");
        boolean isMac = System.getProperty("os.name").toLowerCase(Locale.ROOT).startsWith("mac");
        boolean isLinux = System.getProperty("os.name").toLowerCase(Locale.ROOT).startsWith("linux");

        OperatingSystem os;
        if (isWindows)
            os = OperatingSystems.Windows;
        else if (isMac)
            os = OperatingSystems.macOS;
        else if (isLinux)
            os = OperatingSystems.Linux;
        else
            os = OperatingSystems.UNKNOWN;

        String cpuArch = System.getProperty("os.arch");
        if (cpuArch == null)
            throw new NullPointerException("The os.arch system property is not available.");
        var cpu = CPUArchitecture.fromOrUnknown(cpuArch);

        if (os == OperatingSystems.Linux) {
            CLibrary cLib = CLibraries.GLIBC;
            if (Files.exists(Path.of("/lib/libc.musl-x86_64.so.1")))
                cLib = CLibraries.MUSLC;
            return LinuxMachine.of(cpu, cLib);
        } else {
            return Machine.of(os, cpu);
        }
    }
}
