package dev.hydraulic.types.machines;

import java.util.Locale;

/**
 * <p>A type that represents an operating system or family of closely related operating systems.</p>
 *
 * <p>This interface is implemented by the members of {@link OperatingSystems}. When writing code that wants to represent an operating system
 * use this marker type instead of the values of {@link OperatingSystems} directly so others can create objects representing otherwise
 * unknown operating systems and pass them into your code.</p>
 *
 * <p>This interface doesn't add any additional properties over the {@link MachineCompatibilityAspect} interface.</p>
 *
 * <p>Compatibility note: don't implement this interface unless you're willing to tolerate source breakage. Additional properties useful for
 * identifying operating systems may be added here over time.</p>
 */
public interface OperatingSystem extends MachineCompatibilityAspect {

    /**
     * Lower-cases the name parameter and then looks it up in the {@link OperatingSystems#ALIAS_MAP}.
     *
     * @param name The primary name or alias of an operating system. Case-insensitive.
     * @throws IllegalArgumentException if name is blank.
     * @return a member of the enum or null if not recognized.
     */
    static OperatingSystem fromOrNull(String name) {
        if (name.isBlank())
            throw new IllegalArgumentException("name is blank");
        return OperatingSystems.ALIAS_MAP.get(name.toLowerCase(Locale.ROOT));
    }

    /**
     * Same as {@link #from} but returns {@link OperatingSystems#UNKNOWN} instead of throwing an exception if the name isn't recognized.
     *
     * @param name primary name or alias of the well known operating system.
     */
    static OperatingSystem fromOrUnknown(String name) {
        var os = fromOrNull(name);
        return os != null ? os : OperatingSystems.UNKNOWN;
    }

    /**
     * Looks up an operating system in the {@link OperatingSystems#ALIAS_MAP} or throws {@link UnrecognizedAliasException}.
     *
     * @param name primary name or alias of the well known operating system. Case-insensitive.
     * @throws IllegalArgumentException if name is blank.
     * @throws UnrecognizedAliasException if name isn't a known operating system name or alias (compared case insensitively).
     * @return the operating system entry.
     */
    static OperatingSystem from(String name) {
        var os = fromOrNull(name);
        if (os == null)
            throw new UnrecognizedAliasException(name, OperatingSystems.ALIAS_MAP.keySet());
        return os;
    }
}
