package dev.hydraulic.types.machines;

import java.util.Locale;

/**
 * <p>A type that models a CPU architecture.</p>
 *
 * <p>If you wish, you can extend this to model additional attributes of a CPU ISA such as instruction set extensions.</p>
 */
public interface CPUArchitecture extends MachineCompatibilityAspect {
    /**
     * Lower-cases the name parameter and then looks it up in the {@link CPUArchitectures#ALIAS_MAP alias map}.
     *
     * @param name The primary name or alias of an operating system. Case-insensitive.
     * @return a member of the enum or null if not recognized.
     * @throws IllegalArgumentException if name is blank.
     */
    static CPUArchitecture fromOrNull(String name) {
        if (name.isBlank())
            throw new IllegalArgumentException("name is blank");
        return CPUArchitectures.ALIAS_MAP.get(name.toLowerCase(Locale.ROOT));
    }

    /**
     * Same as {@link #from} but returns {@link OperatingSystems#UNKNOWN} instead of throwing an exception if the name isn't recognized.
     *
     * @param name primary name or alias of the well known operating system.
     */
    static CPUArchitecture fromOrUnknown(String name) {
        var cpu = fromOrNull(name);
        return cpu != null ? cpu : CPUArchitectures.UNKNOWN;
    }

    /**
     * Returns a C library from the {@link CPUArchitectures#ALIAS_MAP alias map} or throws {@link UnrecognizedAliasException}.
     *
     * @param name primary name or alias of the well known operating system. Case-insensitive.
     * @return the C library enum value.
     * @throws IllegalArgumentException   if name is blank.
     * @throws UnrecognizedAliasException if name isn't a known operating system name or alias (compared case insensitively).
     */
    static CPUArchitecture from(String name) {
        var lib = fromOrNull(name);
        if (lib == null)
            throw new UnrecognizedAliasException(name, CPUArchitectures.ALIAS_MAP.keySet());
        return lib;
    }
}
