package dev.hydraulic.types.machines;

import java.util.*;

import static java.util.Set.of;

/**
 * <p>An enumeration of well known, actively used operating systems.</p>
 *
 * <p>This enum doesn't express information about sub-flavors of each OS, for example, it doesn't express what version, distribution,
 * CPU architecture or (for Linux) C library is in use. To include these things use the {@code Machine} type instead.</p>
 *
 * <p>The enum names here prefer to match the capitalization of the brand name rather than the capitalization rules for normal Java
 * identifiers.</p>
 *
 * <p>For ChromeOS and Android the vendor is listed as Google, even though technically the vendor could also be considered as the
 * hardware vendor.</p>
 */
public enum OperatingSystems implements OperatingSystem {
    /** Any desktop/server/embedded Linux distribution that isn't more accurately described by one of the other entries. */
    Linux("Open source", Collections.singleton("linux")),

    /**
     * Apple macOS 10+. The identifier is {@code mac} and aliases are {@code macos}, {@code osx} and {@code darwin}.
     * Although a pure Darwin system isn't technically the same as macOS, in practice nobody runs pure Darwin and thus it's always used as
     * an alias for macOS.
     */
    macOS("Apple", of("mac", "macos", "osx", "darwin")),

    /** Microsoft Windows. Aliases are {@code win}, {@code win32} and {@code win64}. */
    Windows("Microsoft", of("windows", "win", "win32", "win64")),

    /**
     * Any OS that is able to legally use the Android trademark, meaning it either has passed the compliance tests or would definitely
     * do if it were allowed. The vendor is "Google", not "Open source" or the actual distribution vendor and there are no aliases.
     */
    Android("Google", of("android")),

    /** Apple's iconic mobile operating system. Also known as {@code iphoneos}. */
    iOS("Apple", of("ios", "iphoneos")),

    /** Apple TV's operating system. No aliases. */
    tvOS("Apple", of("tvos")),

    /** Apple Watch's operating system. No aliases. */
    watchOS("Apple", of("watchos")),

    /** FreeBSD. */
    FreeBSD("Open source", of("freebsd")),

    /** NetBSD */
    NetBSD("Open source", of("netbsd")),

    /** OpenBSD */
    OpenBSD("Open source", of("openbsd")),

    /** Google ChromeOS. The vendor is "Google" not "Open source". Also known as {@code cros}. */
    ChromeOS("Google", of("chromeos", "cros")),

    /** <a href="https://blackberry.qnx.com/">BlackBerry QNX</a>. */
    QNX("BlackBerry", of("qnx")),

    /** Solaris and any derivatives. The vendor is "Open source". Also known as {@code illumos}. */
    Solaris("Open source", of("solaris", "illumos")),

    /** HP Unix. Also known as {@code hp/ux}. */
    HPUX("HP", of("hpux", "hp/ux")),

    /** IBM AIX. */
    AIX("IBM", of("aix")),   // Still in use, see: https://www.networkworld.com/article/3339381/the-long-slow-death-of-unix.html

    /** IBM zOS, a mainframe operating system. Also known as {@code z/os}. */
    zOS("IBM", of("zos", "z/os")),  // Still in use, see: https://en.wikipedia.org/wiki/Usage_share_of_operating_systems#Decline

    /** Any other operating system that isn't known by this list. The vendor is "Unknown" and it's aliased by "unknown" and "other". */
    UNKNOWN("Unknown", of("other", "unknown"));

    private final String vendor;
    private final Set<String> aliases;

    OperatingSystems(String vendor, Set<String> aliases) {
        this.vendor = vendor;
        this.aliases = aliases;
    }

    @Override
    public String getDisplayName() {
        switch (this) {
            case HPUX:
                return "HP/UX";
            case zOS:
                return "z/OS";
            default:
                return name();
        }
    }

    @Override
    public String getIdentifier() {
        if (this == macOS)
            return "mac";
        else
            return name().toLowerCase(Locale.ROOT);
    }

    @Override
    public Set<String> getAliases() {
        return aliases;
    }

    @Override
    public String getVendor() {
        return vendor;
    }

    /**
     * An unmodifiable map of lower case names and aliases of well known operating systems.
     */
    public final static Map<String, OperatingSystem> ALIAS_MAP;

    static {
        var map = new HashMap<String, OperatingSystems>();
        for (var value : values()) {
            for (var alias : value.aliases) {
                map.put(alias, value);
            }
        }
        ALIAS_MAP = Collections.unmodifiableMap(map);
    }

    /**
     * Various groupings of well known operating systems useful for compatibility testing purposes.
     */
    public static class Families {
        private Families() {}

        /**
         * Returns a set containing all the known operating systems that are generally considered to be source compatible with UNIX
         * operating systems. It isn't related to whether something is formally UNIX certified, has a POSIX compliance layer (e.g. Cygwin)
         * or uses a UNIX core. Therefore, Linux and macOS are considered UNIX but Android, ChromeOS and iOS are not.
         */
        public static Set<OperatingSystem> unix() {
            return java.util.Set.of(Linux, FreeBSD, NetBSD, OpenBSD, Solaris, HPUX, AIX, macOS);
        }

        /**
         * Returns a set containing all the known operating systems that are created by Apple. These operating systems are not necessarily
         * source compatible with each other, but they do all derive from a common NeXTStep root.
         */
        public static Set<OperatingSystem> apple() {
            return java.util.Set.of(macOS, watchOS, tvOS, iOS);
        }

        /**
         * Returns a set containing all the known operating systems that use a Linux kernel, regardless of whether you can compile normal
         * Linux software on them. This currently includes Linux, ChromeOS and Android.
         */
        public static Set<OperatingSystem> linuxKernelBased() {
            return java.util.Set.of(Linux, ChromeOS, Android);
        }
    }
}
